#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion

namespace gov.va.med.vbecs.BOL.UnitTests
{

	
	
 
	/// <summary>
	/// Summary description for VbecsReport.
	/// </summary>
	[TestFixture]
	public class BOL_VbecsReport : BaseTest 
	{
		private BOL.VbecsReport _vbecsReport;
		private BOL.VbecsReport _vbecsReportScheduled;
		private DateTime _currentDate;
		private string _printerName;

		[SetUp]
		protected void SetUp() 
		{
			this.BaseSetUp();
			
			_currentDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			_vbecsReport = new BOL.CTRatioReport();

			_vbecsReport.SetInitialBrokenRules("DlgCTRatioReportCriteria");
			
			_vbecsReportScheduled = new BOL.CTRatioReport(_currentDate.AddDays(-365),_currentDate,true,true);
			
			_printerName = BOL.VbecsReport.GetPrinters()[0].ToString();

			_vbecsReportScheduled.PrinterName = _printerName;
			_vbecsReportScheduled.SetInitialBrokenRules("DlgTransfusionEffectivenessReportCriteria");
		}	

		#region constructors

		[Test]
		public void Ctor_Pass()
		{
			Assert.IsTrue(_vbecsReport != null);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void Ctor_Fail()
		{
		}

		[Test]
		public void Ctor_Parameters_Pass()
		{			
			Assert.IsTrue(_vbecsReportScheduled != null);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void Ctor_Parameters_Fail()
		{
		}

		#endregion

		#region properties

		[Test]
		public void StartDate_Get()
		{
			Assert.AreEqual(_currentDate.Date.AddDays(-1),_vbecsReport.StartDate);
			Assert.AreEqual(_currentDate.Date.AddDays(-365),_vbecsReportScheduled.StartDate);
		}

		[Test]
		public void StartDate_Pass()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			_vbecsReport.StartDate = _currentDate.AddDays(1);
			brokenRulesBefore = _vbecsReport.BrokenRulesCount;
			_vbecsReport.StartDate = _currentDate.AddDays(-1);
			brokenRulesAfter = _vbecsReport.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter<brokenRulesBefore);
		}

		[Test]
		public void StartDate_Fail()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			_vbecsReport.StartDate = _currentDate.AddDays(-1);
			brokenRulesBefore = _vbecsReport.BrokenRulesCount;
			_vbecsReport.StartDate = _currentDate;
			brokenRulesAfter = _vbecsReport.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter>brokenRulesBefore);
		}

		[Test]
		public void EndDate_Get()
		{
			System.DateTime dtCalculated = _currentDate.Date.AddDays(-1).AddHours(23).AddMinutes(59).AddSeconds(59);

			Assert.AreEqual(dtCalculated, _vbecsReport.EndDate, "EndDate");
			Assert.AreEqual(_currentDate.Date.AddHours(23).AddMinutes(59).AddSeconds(59),_vbecsReportScheduled.EndDate);
		}

		[Test]
		public void EndDate_Pass()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			_vbecsReport.EndDate = _currentDate.AddYears(-2);
			brokenRulesBefore = _vbecsReport.BrokenRulesCount;
			_vbecsReport.EndDate = _currentDate.AddDays(-1);
			brokenRulesAfter = _vbecsReport.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter<brokenRulesBefore);
		}

		[Test]
		public void EndDate_StartDate_Fail()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			_vbecsReport.EndDate = _currentDate;
			brokenRulesBefore = _vbecsReport.BrokenRulesCount;
			_vbecsReport.EndDate = _currentDate.AddDays(-2);
			brokenRulesAfter = _vbecsReport.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter>brokenRulesBefore);
		}

		[Test]
		public void EndDate_PrintDate_Fail()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			_vbecsReportScheduled.PrintDate = _currentDate.AddDays(1);
			_vbecsReportScheduled.EndDate = _currentDate.AddDays(-1);
			brokenRulesBefore = _vbecsReportScheduled.BrokenRulesCount;
			_vbecsReportScheduled.EndDate = _currentDate.AddDays(1);
			brokenRulesAfter = _vbecsReportScheduled.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter>brokenRulesBefore);
		}

		[Test]
		public void PrintDate_Get_Pass()
		{
			Assert.AreEqual(_currentDate.Date,_vbecsReportScheduled.PrintDate.Date);
		}

		[Test]
		[ExpectedException( typeof(BOL.BusinessObjectException) )]
		public void PrintDate_Get_Fail()
		{
			Assert.AreEqual(_currentDate.Date,_vbecsReport.PrintDate.Date);
		}

		[Test]
		public void PrintDate_Set_Pass()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			_vbecsReportScheduled.PrintDate = _currentDate.AddDays(-1);
			brokenRulesBefore = _vbecsReportScheduled.BrokenRulesCount;
			_vbecsReportScheduled.PrintDate = _currentDate.AddDays(1);
			brokenRulesAfter = _vbecsReportScheduled.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter<brokenRulesBefore);
		}

		[Test]
		public void PrintDate_Set_Fail()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			_vbecsReportScheduled.PrintDate = _currentDate.AddDays(5);
			brokenRulesBefore = _vbecsReportScheduled.BrokenRulesCount;
			_vbecsReportScheduled.PrintDate = _currentDate.AddMinutes(-1);
			brokenRulesAfter = _vbecsReportScheduled.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter>brokenRulesBefore);
		}

		[Test]
		public void IsScheduled_Get()
		{
			Assert.AreEqual(false,_vbecsReport.IsScheduled);
			Assert.IsTrue(_vbecsReportScheduled.IsScheduled);
		}

		[Test]
		public void IsScheduled_Pass()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			_vbecsReportScheduled.PrintDate = _currentDate.AddMinutes(-1);
			brokenRulesBefore = _vbecsReportScheduled.BrokenRulesCount;
			_vbecsReportScheduled.IsScheduled = false;
			brokenRulesAfter = _vbecsReportScheduled.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter<brokenRulesBefore);
		}

		[Test]
		public void IsScheduled_Fail()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			brokenRulesBefore = _vbecsReport.BrokenRulesCount;
			_vbecsReport.IsScheduled = true;
			brokenRulesAfter = _vbecsReport.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter>brokenRulesBefore);
		}

		[Test]
		public void IsMultiDivisional_Get()
		{
			Assert.AreEqual(false,_vbecsReport.IsMultiDivisional);
			Assert.IsTrue(_vbecsReportScheduled.IsMultiDivisional);
		}

		[Test]
		public void IsMultiDivisional_Pass()
		{
			_vbecsReport.IsMultiDivisional = true;
			Assert.IsTrue(_vbecsReport.IsMultiDivisional);
		}

		[Test]
		[Ignore ( StandardIgnoreReasons.NoFailureConditions )]
		public void IsMultiDivisional_Fail()
		{
		}

		[Test]
		public void IsPreliminary_Get()
		{
			Assert.AreEqual(false,_vbecsReport.IsPreliminary);
			Assert.IsTrue(_vbecsReportScheduled.IsPreliminary);
		}

		[Test]
		[Ignore ( StandardIgnoreReasons.ReadOnlyProperty )]
		public void IsPreliminary_Pass()
		{
		}

		[Test]
		public void PrinterName_Get_Pass()
		{
			Assert.AreEqual(_printerName,_vbecsReportScheduled.PrinterName);
		}

		[Test]
		[ExpectedException( typeof(BOL.BusinessObjectException) )]
		public void PrinterName_Get_Fail()
		{
			Assert.IsNull(_vbecsReport.PrinterName);
		}

		[Test]
		public void PrinterName_Set_Pass()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			_vbecsReportScheduled.PrinterName = "";
			brokenRulesBefore = _vbecsReportScheduled.BrokenRulesCount;
			_vbecsReportScheduled.PrinterName = _printerName;
			brokenRulesAfter = _vbecsReportScheduled.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter<brokenRulesBefore);
		}

		[Test]
		public void PrinterName_Set_NoPrinter_Fail()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			brokenRulesBefore = _vbecsReportScheduled.BrokenRulesCount;
			_vbecsReportScheduled.PrinterName = "     ";
			brokenRulesAfter = _vbecsReportScheduled.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter>brokenRulesBefore);
		}

		[Test]
		[ExpectedException( typeof(BOL.BusinessObjectException) )]
		public void PrinterName_Set_BadPrinter_Fail()
		{
			int brokenRulesBefore;
			int brokenRulesAfter;

			_vbecsReportScheduled.PrinterName = "";
			brokenRulesBefore = _vbecsReportScheduled.BrokenRulesCount;
			_vbecsReportScheduled.PrinterName = "XXXXXXXXXXXXXXX";
			brokenRulesAfter = _vbecsReportScheduled.BrokenRulesCount;
			Assert.IsTrue(brokenRulesAfter<brokenRulesBefore);
		}

		[Test]
		public void IsScheduledReportGreaterThanThirtyDays_Get()
		{
			Assert.AreEqual(false,_vbecsReport.IsScheduledReportGreaterThanThirtyDays);
			Assert.IsTrue(_vbecsReportScheduled.IsScheduledReportGreaterThanThirtyDays);
		}

		[Test]
		[Ignore ( StandardIgnoreReasons.ReadOnlyProperty )]
		public void IsScheduledReportGreaterThanThirtyDays_Pass()
		{
		}

		[Test]
		public void ParameterRow_Pass()
		{
			Assert.IsNotNull(this._vbecsReport.ParameterRow);
		}

		[Test]
		[Ignore ( StandardIgnoreReasons.NoFailureConditions)]
		public void ParameterRow_Fail()
		{
		}

		[Test]
		public void ReportName_Pass()
		{
			this._vbecsReport.ReportName = "REPORTNAME";
			Assert.AreEqual(this._vbecsReport.ReportName, "REPORTNAME");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ReportName_Fail()
		{
		}

		[Test]
		public void DivisionCode_Pass()
		{
			this._vbecsReport.DivisionCode = "DIVISIONCODE";
			Assert.AreEqual(this._vbecsReport.DivisionCode, "DIVISIONCODE");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void DivisionCode_Fail()
		{
		}

		[Test]
		public void SerialFilePath_Pass()
		{
			Assert.IsNotNull(this._vbecsReport.SerialFilePath);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SerialFilePath_Fail()
		{
		}

		[Test]
		public void PrintedBy_Pass()
		{
			this._vbecsReport.PrintedBy = "USERNAME";
			Assert.AreEqual(this._vbecsReport.PrintedBy, "USERNAME");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void PrintedBy_Fail()
		{
		}

		#endregion

		#region Report Scheduler
		
		[Test]
		public void GetScheduledReports_Pass()
		{
			//Schedule a report
			_vbecsReportScheduled.PrintDate = DateTime.Now;

			//Hopefully, this will have a nice default value
			//_vbecsReportScheduled.PrinterName = "SD&D LJ8150DN";
			_vbecsReportScheduled.Schedule();
			//
			//Check that it can be retrieved
			DataTable dtScheduled = BOL.VbecsReport.GetScheduledReports(Common.TaskStatus.NotStarted);
			Assert.IsTrue(dtScheduled.Rows.Count>0,"Rows.Count");
			//
			//Now mark it as printed, so that it won't really print!
			foreach(DataRow dr in dtScheduled.Rows)
			{
				BOL.VbecsReport.UpdateScheduledReportStatus((Guid)dr[TABLES.ScheduledReport.ScheduledReportGuid],Common.TaskStatus.Completed,(byte[])dr[TABLES.ScheduledReport.RowVersion]);
			}
		}

		[Test]
		public void GetScheduledReports_Fail()
		{
			DataTable dtScheduled = BOL.VbecsReport.GetScheduledReports(Common.TaskStatus.Unknown);
			Assert.AreEqual(0,dtScheduled.Rows.Count,"Rows.Count");
		}
				
		[Test]
		public void GetScheduledReport_Pass()
		{
			//Schedule a report
			_vbecsReportScheduled.PrintDate = DateTime.Now;
			//Hopefully, this will have a nice default value
			//_vbecsReportScheduled.PrinterName = "SD&D LJ8150DN";
			_vbecsReportScheduled.Schedule();
			//
			//Check that it can be retrieved
			DataRow dr = BOL.VbecsReport.GetScheduledReports(Common.TaskStatus.NotStarted).Rows[0];
			DataTable dt = BOL.VbecsReport.GetScheduledReport((Guid)dr[TABLES.ScheduledReport.ScheduledReportGuid]);
			Assert.AreEqual(1,dt.Rows.Count,"Rows.Count");
			//
			//Now mark it as printed, so that it won't really print!
			BOL.VbecsReport.UpdateScheduledReportStatus((Guid)dr[TABLES.ScheduledReport.ScheduledReportGuid],Common.TaskStatus.Completed,(byte[])dr[TABLES.ScheduledReport.RowVersion]);
		}

		[Test]
		public void GetScheduledReport_Fail()
		{
			DataTable dt = BOL.VbecsReport.GetScheduledReport(Guid.NewGuid());
			Assert.AreEqual(0,dt.Rows.Count,"Rows.Count");
		}

		[Test]
		public void UpdateScheduledReportStatus_Pass()
		{
			//Schedule a report
			_vbecsReportScheduled.PrintDate = DateTime.Now;
			//Hopefully, this will have a nice default value
			//_vbecsReportScheduled.PrinterName = "SD&D LJ8150DN";
			_vbecsReportScheduled.Schedule();
			//
			//Now mark it as printed, so that it won't really print!
			DataTable dtScheduled = BOL.VbecsReport.GetScheduledReports(Common.TaskStatus.NotStarted);
			foreach(DataRow dr in dtScheduled.Rows)
			{
				Assert.IsTrue(BOL.VbecsReport.UpdateScheduledReportStatus((Guid)dr[TABLES.ScheduledReport.ScheduledReportGuid],Common.TaskStatus.InProgress,(byte[])dr[TABLES.ScheduledReport.RowVersion]),"UpdateReportStatus");
			}		
		}

		[Test]
		[ExpectedException(typeof(Common.RowVersionException))]
		public void UpdateScheduledReportStatus_Fail()
		{
			Assert.IsTrue(!BOL.VbecsReport.UpdateScheduledReportStatus(Guid.NewGuid(),Common.TaskStatus.InProgress,new byte[8]),"InvalidGuid");
		}

		[Test]
		public void InsertScheduledReport_Pass()
		{
			_vbecsReportScheduled.PrintDate = DateTime.Now;
			//Hopefully, this will have a nice default value
			//_vbecsReportScheduled.PrinterName = "SD&D LJ8150DN";
			Assert.IsTrue(_vbecsReportScheduled.InsertScheduledReport(),"InsertScheduledReport");
			//
			//Now mark it as printed, so that it won't really print!
			DataTable dtScheduled = BOL.VbecsReport.GetScheduledReports(Common.TaskStatus.NotStarted);
			foreach(DataRow dr in dtScheduled.Rows)
			{
				BOL.VbecsReport.UpdateScheduledReportStatus((Guid)dr[TABLES.ScheduledReport.ScheduledReportGuid],Common.TaskStatus.Completed,(byte[])dr[TABLES.ScheduledReport.RowVersion]);
			}
		}

		[Test]
		[Ignore ( StandardIgnoreReasons.NoFailureConditions )]
		public void InsertScheduledReport_Fail()
		{
		}

		#endregion

		#region methods



		[Test]
		public void GetPrinters_Pass()
		{
			Assert.IsNotNull(BOL.VbecsReport.GetPrinters());
		}

		[Test]
		[Ignore ( StandardIgnoreReasons.NoFailureConditions )]
		public void GetPrinters_Fail()
		{
		}

		[Test]
		public void GetReportHeader_Pass()
		{
			Assert.IsNotNull(BOL.VbecsReport.GetReportHeader("My Name here"));
		}

		[Test]
		[ExpectedException(typeof(Common.DataAccessLayerException))]
		public void GetReportHeader_Fail()
		{
			Common.LogonUser.LogonUserDivisionCode = "XXXXX";
			Assert.IsNotNull(BOL.VbecsReport.GetReportHeader("My Name here"));
		}

		[Test]
		public void GetReportHeader_Division_Pass()
		{
			Assert.IsNotNull(BOL.VbecsReport.GetReportHeader("My Name here",UnitTestConfig.TestDivisionCode));
		}

		[Test]
		[ExpectedException(typeof(Common.DataAccessLayerException))]
		public void GetReportHeader_Division_Fail()
		{
			Assert.IsNotNull(BOL.VbecsReport.GetReportHeader("My Name here","XXXXX"));
		}

		[Test]
		public void GetReportFooter_Pass()
		{
			Assert.IsNotNull(BOL.VbecsReport.GetReportFooter("Blah!"));
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetReportFooter_Fail()
		{
		}

		[Test]
		public void IsPrinterConfigured_Pass()
		{
			Assert.IsTrue(BOL.VbecsReport.IsPrinterConfigured(_printerName));
		}

		[Test]
		public void IsPrinterConfigured_Fail()
		{
			Assert.IsTrue(!BOL.VbecsReport.IsPrinterConfigured(null));
		}

		[Test]
		public void MultiDivisionalSort_Pass()
		{
			DataTable dtUnsorted = new DataTable("UnsortedTable");
			dtUnsorted.Columns.Add(Common.VbecsTables.VamcDivision.DivisionCode);
			dtUnsorted.Columns.Add(Common.VbecsTables.VamcDivision.DivisionName);
			//
			DataRow dr = dtUnsorted.NewRow();
			dr[Common.VbecsTables.VamcDivision.DivisionCode] = "XXXX";
			dr[Common.VbecsTables.VamcDivision.DivisionName] = "XXXX";
			dtUnsorted.Rows.Add(dr);
			//
			dr = dtUnsorted.NewRow();
			dr[Common.VbecsTables.VamcDivision.DivisionCode] = UnitTestConfig.TestDivisionCode;
			dr[Common.VbecsTables.VamcDivision.DivisionName] = UnitTestConfig.TestDivisionName;
			dtUnsorted.Rows.Add(dr);
			//
			DataTable dtSorted = BOL.VbecsReport.MultiDivisionalSort(dtUnsorted);
			Assert.AreEqual(dtUnsorted.Rows[1][Common.VbecsTables.VamcDivision.DivisionName].ToString(),dtSorted.Rows[0][Common.VbecsTables.VamcDivision.DivisionName].ToString());
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void MultiDivisionalSort_Fail()
		{
			DataTable dtUnsorted = new DataTable("UnsortedTable");
			//
			DataRow dr = dtUnsorted.NewRow();
			dtUnsorted.Rows.Add(dr);
			//
			dr = dtUnsorted.NewRow();
			dtUnsorted.Rows.Add(dr);
			//
			DataTable dtSorted = BOL.VbecsReport.MultiDivisionalSort(dtUnsorted);
		}

		[Test]
		public void LoadPropertiesFromDataRow_Pass()
		{
			BOL.VbecsReport rpt=new BOL.TransfusionEffectivenessReport();

			DataRow dr=rpt.ParameterRow.Table.NewRow();

			dr["IsMultidivisional"]=false;
			dr["DivisionCode"]=UnitTestConfig.TestDivisionCode;
			dr["EndDate"]=DateTime.Today;
			dr["PrintDate"]=DateTime.Today;
			dr["PrintedBy"]=Common.LogonUser.LogonUserName;
			dr["ReportName"]="Report";
			dr["StartDate"]=DateTime.Today.AddYears(-1);

			rpt.LoadPropertiesFromDataRow(dr);
			
			rpt.IsScheduled=true;

			Assert.AreEqual(rpt.IsMultiDivisional, false);
			Assert.AreEqual(rpt.DivisionCode, UnitTestConfig.TestDivisionCode);
			Assert.AreEqual(rpt.EndDate, DateTime.Today);
			Assert.AreEqual(rpt.PrintDate, DateTime.Today);
			Assert.AreEqual(rpt.PrintedBy, Common.LogonUser.LogonUserName);
			Assert.AreEqual(rpt.ReportName, "Report");
			Assert.AreEqual(rpt.StartDate, DateTime.Today.AddYears(-1));

		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadPropertiesFromDataRow_Fail()
		{
			BOL.VbecsReport rpt=new BOL.TransfusionEffectivenessReport();

			DataRow dr=BOL.ReferenceData.GetProductModification("F");

			rpt.LoadPropertiesFromDataRow(dr);
		}

		[Test]
		public void SaveParameters_Pass()
		{
			_vbecsReportScheduled.PrintDate = DateTime.Now;
			//Hopefully, this will have a nice default value
			//_vbecsReportScheduled.PrinterName = "SD&D LJ8150DN";
			_vbecsReportScheduled.SaveParameters();
			//
			//Check that it can be retrieved
			DataTable dtScheduled = BOL.VbecsReport.GetScheduledReports(Common.TaskStatus.NotStarted);
			Assert.IsTrue(dtScheduled.Rows.Count>0,"Rows.Count");
			//
			//Now mark it as printed, so that it won't really print!
			foreach(DataRow dr in dtScheduled.Rows)
			{
				BOL.VbecsReport.UpdateScheduledReportStatus((Guid)dr[TABLES.ScheduledReport.ScheduledReportGuid],Common.TaskStatus.Completed,(byte[])dr[TABLES.ScheduledReport.RowVersion]);
			}
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SaveParameters_Fail()
		{
		}

		[Test]
		public void Schedule_Pass()
		{
			_vbecsReportScheduled.PrintDate = DateTime.Now;
			//Hopefully, this will have a nice default value
			//_vbecsReportScheduled.PrinterName = "SD&D LJ8150DN";
			_vbecsReportScheduled.Schedule();
			//
			//Check that it can be retrieved
			DataTable dtScheduled = BOL.VbecsReport.GetScheduledReports(Common.TaskStatus.NotStarted);
			Assert.IsTrue(dtScheduled.Rows.Count>0,"Rows.Count");
			//
			//Now mark it as printed, so that it won't really print!
			foreach(DataRow dr in dtScheduled.Rows)
			{
				BOL.VbecsReport.UpdateScheduledReportStatus((Guid)dr[TABLES.ScheduledReport.ScheduledReportGuid],Common.TaskStatus.Completed,(byte[])dr[TABLES.ScheduledReport.RowVersion]);
			}
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Schedule_Fail()
		{
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			DataRow test = new DataTable().NewRow();
			//
			Assert.AreEqual ( this._vbecsReport.LoadDataRowFromThis( test ), test );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}

		#endregion
	}
}

#endif
